import React, { useState, useEffect } from 'react';
import { Home, FileText, Users, MoreHorizontal, Plus, ArrowRight } from 'lucide-react';
import { BrowserRouter as Router, Routes, Route, useNavigate, useLocation } from 'react-router-dom';
import { motion, AnimatePresence } from 'framer-motion';
import { getDashboard, getInvoices, getCustomers } from './services/api';

// Dummy Components for now
const Dashboard = () => {
  const [data, setData] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    getDashboard()
      .then(setData)
      .finally(() => setLoading(false));
  }, []);

  if (loading) return <div className="content-padding">Loading Dashboard...</div>;

  return (
    <div className="content-padding animate-slide-up">
      <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '24px' }}>
        <h1>Dashboard</h1>
        <div className="icon-badge">🔔</div>
      </div>

      <div className="card" style={{ background: 'linear-gradient(135deg, #0077c5 0%, #00a1ff 100%)', color: 'white' }}>
        <p style={{ opacity: 0.8, fontSize: '14px' }}>Cash Flow</p>
        <h2 style={{ fontSize: '32px', color: 'white', margin: '8px 0' }}>${data.totalInvoiced.toLocaleString()}</h2>
        <div style={{ height: '100px', width: '100%', background: 'rgba(255,255,255,0.1)', borderRadius: '8px', marginTop: '16px', display: 'flex', alignItems: 'flex-end', padding: '10px', gap: '8px' }}>
          {data.cashFlow.map((h, i) => (
            <div key={i} style={{ flex: 1, height: `${h}%`, background: 'white', borderRadius: '4px 4px 0 0', opacity: 0.8 }} />
          ))}
        </div>
      </div>

      <div style={{ display: 'flex', gap: '12px', marginBottom: '24px' }}>
        <div className="card" style={{ flex: 1, marginBottom: 0 }}>
          <p style={{ fontSize: '12px', color: 'var(--text-muted)' }}>Invoices</p>
          <h3 style={{ color: 'var(--primary)' }}>${(data.totalInvoiced - data.overdue).toLocaleString()}</h3>
        </div>
        <div className="card" style={{ flex: 1, marginBottom: 0 }}>
          <p style={{ fontSize: '12px', color: 'var(--text-muted)' }}>Overdue</p>
          <h3 style={{ color: 'var(--error)' }}>${data.overdue.toLocaleString()}</h3>
        </div>
      </div>

      <h4>Recent Activity</h4>
      {data.recentActivity.map(item => (
        <div key={item.id} className="card" style={{ marginTop: '12px' }}>
          <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
            <div>
              <p style={{ fontWeight: 600 }}>{item.title}</p>
              <p style={{ fontSize: '12px', color: 'var(--text-muted)' }}>{item.subtitle}</p>
            </div>
            <p style={{ fontWeight: 700, color: `var(--${item.type})` }}>+${item.amount}</p>
          </div>
        </div>
      ))}
    </div>
  );
};

import InvoiceForm from './components/InvoiceForm';

const Invoices = () => {
  const [showForm, setShowForm] = useState(false);
  const [invoices, setInvoices] = useState([]);
  const [loading, setLoading] = useState(true);

  const fetchInvoices = () => {
    setLoading(true);
    getInvoices()
      .then(setInvoices)
      .finally(() => setLoading(false));
  };

  useEffect(() => {
    fetchInvoices();
  }, []);

  return (
    <div className="content-padding animate-slide-up">
      <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '24px' }}>
        <h1>Invoices</h1>
        <Plus size={24} color="var(--primary)" />
      </div>

      {loading ? (
        <p>Loading invoices...</p>
      ) : invoices.length === 0 ? (
        <p>No invoices found.</p>
      ) : (
        invoices.map(inv => (
          <div key={inv.id} className="card" style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
            <div>
              <p style={{ fontWeight: 600 }}>INV-{inv.id.toString().padStart(3, '0')}</p>
              <p style={{ fontSize: '12px', color: 'var(--text-muted)' }}>{inv.customer_name}</p>
            </div>
            <div style={{ textAlign: 'right' }}>
              <p style={{ fontWeight: 700 }}>${parseFloat(inv.total).toFixed(2)}</p>
              <p style={{
                fontSize: '11px',
                padding: '4px 10px',
                borderRadius: '8px',
                background: inv.status === 'Overdue' ? '#ffebee' : inv.status === 'Paid' ? '#e8f5e9' : '#f5f5f5',
                color: inv.status === 'Overdue' ? 'var(--error)' : inv.status === 'Paid' ? 'var(--success)' : 'var(--text-muted)',
                display: 'inline-block',
                fontWeight: 600,
                marginTop: '4px',
                textTransform: 'uppercase',
                letterSpacing: '0.5px'
              }}>
                {inv.status}
              </p>
            </div>
          </div>
        ))
      )}

      <button
        onClick={() => setShowForm(true)}
        className="btn-primary"
        style={{ position: 'fixed', bottom: '100px', right: '20px', borderRadius: '30px', boxShadow: 'var(--shadow-lg)', display: 'flex', alignItems: 'center', gap: '8px' }}
      >
        <Plus size={20} /> New Invoice
      </button>

      <AnimatePresence>
        {showForm && <InvoiceForm onClose={() => { setShowForm(false); fetchInvoices(); }} />}
      </AnimatePresence>
    </div>
  );
};

const Customers = () => {
  const [customers, setCustomers] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    getCustomers()
      .then(setCustomers)
      .finally(() => setLoading(false));
  }, []);

  return (
    <div className="content-padding animate-slide-up">
      <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '24px' }}>
        <h1>Customers</h1>
        <Plus size={24} color="var(--primary)" />
      </div>

      {loading ? (
        <p>Loading customers...</p>
      ) : customers.length === 0 ? (
        <p>No customers found.</p>
      ) : (
        customers.map(c => (
          <div key={c.id} className="card" style={{ display: 'flex', alignItems: 'center', gap: '16px' }}>
            <div style={{ width: '40px', height: '40px', background: 'var(--bg-light)', borderRadius: '20px', display: 'flex', alignItems: 'center', justifyContent: 'center', fontWeight: 600, color: 'var(--primary)' }}>
              {c.name.charAt(0)}
            </div>
            <div>
              <p style={{ fontWeight: 600 }}>{c.name}</p>
              <p style={{ fontSize: '12px', color: 'var(--text-muted)' }}>{c.email}</p>
            </div>
          </div>
        ))
      )}
    </div>
  );
};

const More = ({ user, onLogout }) => (
  <div className="content-padding animate-slide-up">
    <h1>More</h1>

    <div className="card" style={{ marginTop: '24px', display: 'flex', alignItems: 'center', gap: '16px' }}>
      <div style={{ width: '50px', height: '50px', background: 'var(--primary)', borderRadius: '25px', display: 'flex', alignItems: 'center', justifyContent: 'center', color: 'white', fontWeight: 700, fontSize: '20px' }}>
        {user.username.charAt(0)}
      </div>
      <div>
        <p style={{ fontWeight: 600, fontSize: '18px' }}>{user.username}</p>
        <p style={{ fontSize: '14px', color: 'var(--text-muted)' }}>{user.role}</p>
      </div>
    </div>

    <div style={{ marginTop: '32px' }}>
      <button
        onClick={onLogout}
        className="card"
        style={{ width: '100%', textAlign: 'left', border: 'none', cursor: 'pointer', background: 'white', display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}
      >
        <span style={{ color: 'var(--error)', fontWeight: 600 }}>Sign Out</span>
        <ArrowRight size={20} color="var(--error)" />
      </button>
    </div>
  </div>
);

const BottomNav = () => {
  const navigate = useNavigate();
  const location = useLocation();

  const navItems = [
    { name: 'Dashboard', path: '/', icon: Home },
    { name: 'Invoices', path: '/invoices', icon: FileText },
    { name: 'Customers', path: '/customers', icon: Users },
    { name: 'More', path: '/more', icon: MoreHorizontal },
  ];

  return (
    <nav style={{
      position: 'fixed',
      bottom: 0,
      left: '50%',
      transform: 'translateX(-50%)',
      width: '100%',
      maxWidth: '500px',
      height: '80px',
      background: 'rgba(255, 255, 255, 0.8)',
      backdropFilter: 'blur(10px)',
      borderTop: '1px solid var(--border-color)',
      display: 'flex',
      justifyContent: 'space-around',
      alignItems: 'center',
      paddingBottom: '20px',
      zIndex: 1000
    }}>
      {navItems.map((item) => {
        const Icon = item.icon;
        const isActive = location.pathname === item.path;
        return (
          <div
            key={item.name}
            onClick={() => navigate(item.path)}
            style={{
              display: 'flex',
              flexDirection: 'column',
              alignItems: 'center',
              gap: '4px',
              color: isActive ? 'var(--primary)' : 'var(--text-muted)',
              cursor: 'pointer'
            }}
          >
            <Icon size={24} strokeWidth={isActive ? 2.5 : 2} />
            <span style={{ fontSize: '10px', fontWeight: isActive ? 700 : 500 }}>{item.name}</span>
          </div>
        );
      })}
    </nav>
  );
};

import Login from './pages/Login';

function App() {
  const [user, setUser] = useState(null);

  if (!user) {
    return <Login onLogin={setUser} />;
  }

  return (
    <Router>
      <div className="app-container">
        <AnimatePresence mode="wait">
          <Routes>
            <Route path="/" element={<Dashboard />} />
            <Route path="/invoices" element={<Invoices />} />
            <Route path="/customers" element={<Customers />} />
            <Route path="/more" element={<More user={user} onLogout={() => setUser(null)} />} />
          </Routes>
        </AnimatePresence>
        <BottomNav />
      </div>
    </Router>
  );
}

export default App;
